<?php
/**
 * NexaWP Connector
 *
 * Centralized management plugin for WordPress sites via NexaWP Manager
 *
 * @package   NexaWP Connector
 * @author    NexaWP Team
 * @copyright 2023-2025
 * @license   GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:  NexaWP Connector
 * Plugin URI:   https://nexawpmanager.com/
 * Description:  Secure connector for WordPress site management via NexaWP Manager
 * Version:      1.0.7
 * Author:       NexaWP Manager
 * Author URI:   https://nexawpmanager.com/
 * License:      GPL-2.0-or-later
 * License URI:  https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:  nexawp-connector
 * Domain Path:  /languages
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('NEXAWP_VERSION', '1.0.6');
define('NEXAWP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NEXAWP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NEXAWP_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Load the main REST controller
require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-rest-controller.php';

/**
 * Main plugin class
 */
class NexaWP_Connector {
    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * REST controller instance
     */
    private $rest_controller;

    /**
     * Retrieve instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', [$this, 'early_cors_init'], 0);
        add_action('rest_api_init', [$this, 'init_rest_api'], 0);
        add_action('init', [$this, 'init_plugin'], 0);
    }

    /**
     * Early CORS initialization
     */
    public function early_cors_init() {
        if (!$this->rest_controller) {
            $this->load_controllers();
            $this->rest_controller = new NexaWP_REST_Controller();
        }
    }

    /**
     * Initialize REST API
     */
    public function init_rest_api() {
        error_log('[NexaWP] Initializing REST API');
        if (!$this->rest_controller) {
            $this->load_controllers();
            $this->rest_controller = new NexaWP_REST_Controller();
            $this->rest_controller->register_routes();
        }
        error_log('[NexaWP] REST API initialized');
    }

    /**
     * Initialize plugin
     */
    public function init_plugin() {
        // Only load controllers when it's a REST request
        if (defined('REST_REQUEST') && REST_REQUEST) {
            $this->load_controllers();
            $this->rest_controller = new NexaWP_REST_Controller();
        }
        $this->init_hooks();
    }

    /**
     * Load controllers
     */
    private function load_controllers() {
        require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-plugins-controller.php';
        require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-themes-controller.php';
        require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-users-controller.php';
        require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-backup-controller.php';
        require_once NEXAWP_PLUGIN_DIR . 'includes/class-nexawp-core-controller.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('init', [$this, 'load_textdomain'], 1);

        if (is_admin()) {
            add_action('admin_menu', [$this, 'add_admin_menu'], 99);
            add_filter('plugin_action_links_' . NEXAWP_PLUGIN_BASENAME, [$this, 'add_settings_link'], 99);

            if (isset($_GET['page']) && $_GET['page'] === 'nexawp-settings') {
                add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
            }

            add_action('wp_ajax_nexawp_generate_api_key_ajax', [$this, 'handle_generate_api_key_ajax']);
        }

        if (defined('REST_REQUEST') && REST_REQUEST && $this->rest_controller) {
            add_action('rest_api_init', [$this->rest_controller, 'register_routes']);
        }
    }

    /**
     * Load plugin textdomain
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'nexawp-connector',
            false,
            dirname(plugin_basename(__FILE__)) . '/languages'
        );
    }

    /**
     * Add settings page
     */
    public function add_admin_menu() {
        add_submenu_page(
            'options-general.php',
            __('NexaWP Connector', 'nexawp-connector'),
            __('NexaWP Connector', 'nexawp-connector'),
            'manage_options',
            'nexawp-settings',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Add Settings link in plugin list
     */
    public function add_settings_link($links) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('options-general.php?page=nexawp-settings'),
            __('Settings', 'nexawp-connector')
        );
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Enqueue admin assets on settings page
     */
    public function enqueue_admin_assets($hook) {
        if ('settings_page_nexawp-settings' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'nexawp-admin-styles',
            NEXAWP_PLUGIN_URL . 'assets/css/styles.css',
            ['dashicons'],
            NEXAWP_VERSION
        );

        wp_enqueue_script(
            'nexawp-admin-scripts',
            NEXAWP_PLUGIN_URL . 'assets/js/script.js',
            ['jquery'],
            NEXAWP_VERSION,
            true
        );

        wp_localize_script(
            'nexawp-admin-scripts',
            'nexaWPAdmin',
            [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('nexawp-admin-nonce'),
                'i18n'    => [
                    'copied'        => __('Copied!', 'nexawp-connector'),
                    'copyError'     => __('Copy failed!', 'nexawp-connector'),
                    'generating'    => __('Generating...', 'nexawp-connector'),
                    'generateError' => __('Error generating API key', 'nexawp-connector'),
                    'ajaxError'     => __('Failed to generate API key. Please try again.', 'nexawp-connector'),
                ]
            ]
        );
    }

    /**
     * Handle AJAX to generate API key
     */
    public function handle_generate_api_key_ajax() {
        check_ajax_referer('nexawp-admin-nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'nexawp-connector')]);
        }

        try {
            $api_key = wp_generate_password(32, false);
            $current_time = current_time('mysql');

            update_option('nexawp_api_key', $api_key);
            update_option('nexawp_api_key_generated_on', $current_time);
            update_option('nexawp_api_key_displayed', 'no');

            wp_send_json_success([
                'html' => sprintf('
                   <div class="nexawp-copy-field">
                       <input type="text" id="api-key" value="%1$s" readonly>
                       <button class="nexawp-copy-button" data-target="api-key">
                           <span class="dashicons dashicons-admin-page"></span>
                           <span class="tooltip">%2$s</span>
                       </button>
                   </div>
                   <div class="nexawp-key-meta">
                       <span class="dashicons dashicons-calendar"></span>
                       %3$s
                   </div>',
                    esc_attr($api_key),
                    esc_html__('Copy Key', 'nexawp-connector'),
                    esc_html(sprintf(
                        __('Generated on: %s', 'nexawp-connector'),
                        date_i18n(
                            get_option('date_format') . ' ' . get_option('time_format'),
                            strtotime($current_time)
                        )
                    ))
                )
            ]);
        } catch (Exception $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'nexawp-connector'));
        }
        require_once NEXAWP_PLUGIN_DIR . 'admin/settings-page.php';
    }

    /**
     * Activation hook
     */
    public static function activate() {
        do_action('nexawp_activated');
    }

    /**
     * Deactivation hook
     */
    public static function deactivate() {
        do_action('nexawp_deactivated');
    }

    /**
     * Uninstall hook
     */
    public static function uninstall() {
        if (!current_user_can('activate_plugins')) {
            return;
        }
        delete_option('nexawp_api_key');
        delete_option('nexawp_api_key_generated_on');
        delete_option('nexawp_api_key_displayed');
        do_action('nexawp_uninstalled');
    }
}

// Hooks for plugin lifecycle
register_activation_hook(__FILE__, ['NexaWP_Connector', 'activate']);
register_deactivation_hook(__FILE__, ['NexaWP_Connector', 'deactivate']);

// Initialize plugin
add_action('plugins_loaded', ['NexaWP_Connector', 'get_instance'], 99);


/**
 * --------------------------------------------------------
 * Magic Link Auto-Login Logic
 * --------------------------------------------------------
 * This hook detects the 'nexawp_auto_login_token' in the URL.
 * If valid, it logs the user in and redirects to /wp-admin.
 */
add_action('init', 'nexawp_handle_magic_link_login');
function nexawp_handle_magic_link_login() {
    if ( isset($_GET['nexawp_auto_login_token']) ) {
        $token = sanitize_key($_GET['nexawp_auto_login_token']);
        $user_id = get_transient("nexawp_auto_login_$token");

        if ( $user_id ) {
            delete_transient("nexawp_auto_login_$token");
            wp_set_current_user($user_id);
            wp_set_auth_cookie($user_id, true);
            wp_safe_redirect(admin_url());
            exit;
        } else {
            wp_die('Auto-login token invalid or expired.');
        }
    }
}
