<?php
/**
 * NexaWP Users Controller
 *
 * Handles all user-related REST API endpoints
 *
 * @package NexaWP
 */

class NexaWP_Users_Controller {

    /**
     * API Namespace
     *
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Register REST routes for users
     */
    public function register_routes() {
        // Get all users
        register_rest_route($this->namespace, '/users', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_users'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_users_args(),
        ));

        // Get single user
        register_rest_route($this->namespace, '/users/(?P<id>\d+)', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_user'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_user_args(),
        ));

        // Create user
        register_rest_route($this->namespace, '/users', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'create_user'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_create_user_args(),
        ));

        // Update user
        register_rest_route($this->namespace, '/users/(?P<id>\d+)', array(
            'methods'             => WP_REST_Server::EDITABLE,
            'callback'            => array($this, 'update_user'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_update_user_args(),
        ));

        // Delete user
        register_rest_route($this->namespace, '/users/(?P<id>\d+)', array(
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => array($this, 'delete_user'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_user_args(),
        ));

        // Delete bulk user
        register_rest_route($this->namespace, '/users/bulk-delete', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'bulk_delete_users'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_bulk_delete_user_args(),
        ));
    }

    /**
     * Get all users
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_users($request) {
        $args = array(
            'number'   => $request->get_param('per_page') ?: 100, // Increase max users
            'paged'    => $request->get_param('page') ?: 1,
            'role'     => $request->get_param('role'),
            'orderby'  => $request->get_param('orderby') ?: 'login',
            'order'    => $request->get_param('order') ?: 'ASC',
            'search'   => $request->get_param('search'),
        );

        $users_query = new WP_User_Query($args);
        $total_users = $users_query->get_total();
        $users = array();

        foreach ($users_query->get_results() as $user) {
            $users[] = $this->prepare_user_data($user);
        }

        $response = new WP_REST_Response($users, 200);
        $response->header('X-WP-Total', $total_users);
        $response->header('X-WP-TotalPages', ceil($total_users / $args['number']));

        return $response;
    }

    /**
     * Get single user
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_user($request) {
        $user_id = (int) $request->get_param('id');
        $user = get_userdata($user_id);

        if (!$user) {
            return new WP_Error(
                'user_not_found',
                __('User not found', 'nexawp-connector'),
                array('status' => 404)
            );
        }

        return new WP_REST_Response($this->prepare_user_data($user), 200);
    }

    /**
     * Create user
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function create_user($request) {
        $username = sanitize_user($request->get_param('username'));
        $email = sanitize_email($request->get_param('email'));
        $password = $request->get_param('password');
        $role = $request->get_param('role') ?: 'subscriber';
        $first_name = sanitize_text_field($request->get_param('first_name') ?: '');
        $last_name = sanitize_text_field($request->get_param('last_name') ?: '');
        $website = esc_url_raw($request->get_param('website') ?: '');

        // Vérifier si l'utilisateur existe déjà
        if (username_exists($username)) {
            return new WP_Error(
                'username_exists',
                __('Username already exists', 'nexawp-connector'),
                array('status' => 400)
            );
        }

        if (email_exists($email)) {
            return new WP_Error(
                'email_exists',
                __('Email address already exists', 'nexawp-connector'),
                array('status' => 400)
            );
        }

        // Vérification de la force du mot de passe
        if (!$this->is_password_strong($password)) {
            return new WP_Error(
                'weak_password',
                __('Password does not meet strength requirements', 'nexawp-connector'),
                array('status' => 400)
            );
        }

        $user_data = array(
            'user_login' => $username,
            'user_email' => $email,
            'user_pass'  => $password,
            'role'       => $role,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'user_url'   => $website,
        );

        $user_id = wp_insert_user($user_data);

        if (is_wp_error($user_id)) {
            return $user_id;
        }

        $user = get_userdata($user_id);

        return new WP_REST_Response($this->prepare_user_data($user), 201);
    }

    /**
     * Get user arguments
     *
     * @return array
     */
    private function get_user_args() {
        return array(
            'id' => array(
                'required'          => true,
                'type'             => 'integer',
                'sanitize_callback' => 'absint',
            ),
        );
    }

    /**
     * Get create user arguments
     *
     * @return array
     */
    private function get_create_user_args() {
        return array(
            'username' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_user',
            ),
            'email' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_email',
            ),
            'password' => array(
                'required'          => true,
                'type'             => 'string',
            ),
            'role' => array(
                'type'             => 'string',
                'default'          => 'subscriber',
                'enum'             => array('administrator', 'editor', 'author', 'contributor', 'subscriber'),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'first_name' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'last_name' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'website' => array(
                'type'             => 'string',
                'sanitize_callback' => 'esc_url_raw',
            ),
        );
    }

    /**
     * Get update user arguments
     *
     * @return array
     */
    private function get_update_user_args() {
        return array(
            'email' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_email',
            ),
            'password' => array(
                'type'             => 'string',
            ),
            'role' => array(
                'type'             => 'string',
                'enum'             => array('administrator', 'editor', 'author', 'contributor', 'subscriber'),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'first_name' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'last_name' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'website' => array(
                'type'             => 'string',
                'sanitize_callback' => 'esc_url_raw',
            ),
        );
    }

    /**
     * Update user
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_user($request) {
        try {
            $user_id = (int) $request->get_param('id');
            $user = get_userdata($user_id);

            if (!$user) {
                return new WP_Error(
                    'user_not_found',
                    __('User not found', 'nexawp-connector'),
                    array('status' => 404)
                );
            }

            $user_data = array(
                'ID' => $user_id,
            );

            // Journaliser les paramètres reçus pour déboguer
            error_log('[NexaWP] Update user request parameters: ' . print_r($request->get_params(), true));

            // Update email if provided
            if ($request->has_param('email')) {
                $new_email = sanitize_email($request->get_param('email'));
                // Vérifier si l'email existe déjà pour un autre utilisateur
                if ($new_email !== $user->user_email && email_exists($new_email)) {
                    return new WP_Error(
                        'email_exists',
                        __('Email address already exists', 'nexawp-connector'),
                        array('status' => 400)
                    );
                }
                $user_data['user_email'] = $new_email;
            }

            // Update password if provided
            if ($request->has_param('password') && !empty($request->get_param('password'))) {
                $password = $request->get_param('password');

                // Vérification de la force du mot de passe
                if (!$this->is_password_strong($password)) {
                    return new WP_Error(
                        'weak_password',
                        __('Password does not meet strength requirements', 'nexawp-connector'),
                        array('status' => 400)
                    );
                }

                $user_data['user_pass'] = $password;
            }

            // Update first name if provided
            if ($request->has_param('first_name')) {
                $user_data['first_name'] = sanitize_text_field($request->get_param('first_name'));
            }

            // Update last name if provided
            if ($request->has_param('last_name')) {
                $user_data['last_name'] = sanitize_text_field($request->get_param('last_name'));
            }

            // Update website if provided
            if ($request->has_param('website')) {
                $user_data['user_url'] = esc_url_raw($request->get_param('website'));
            }

            // Update role if provided - Accepter tous les rôles valides
            if ($request->has_param('role')) {
                $new_role = sanitize_text_field($request->get_param('role'));

                // Récupérer tous les rôles valides dans WordPress
                global $wp_roles;
                $available_roles = array_keys($wp_roles->roles);

                // Journaliser pour le débogage
                error_log('[NexaWP] Available roles: ' . print_r($available_roles, true));
                error_log('[NexaWP] Requested role: ' . $new_role);

                // Vérifier si le rôle demandé est valide
                if (!in_array($new_role, $available_roles)) {
                    return new WP_Error(
                        'invalid_role',
                        __('The selected role is invalid', 'nexawp-connector'),
                        array('status' => 400)
                    );
                }

                // Vérifier si c'est le dernier administrateur
                if (in_array('administrator', $user->roles) && $new_role !== 'administrator' && $this->is_last_admin($user_id)) {
                    return new WP_Error(
                        'last_admin',
                        __('Cannot change role of the last administrator', 'nexawp-connector'),
                        array('status' => 403)
                    );
                }

                $user->set_role($new_role);
            }

            $updated_user_id = wp_update_user($user_data);

            if (is_wp_error($updated_user_id)) {
                error_log('[NexaWP] Error updating user: ' . $updated_user_id->get_error_message());
                return $updated_user_id;
            }

            $updated_user = get_userdata($updated_user_id);
            error_log('[NexaWP] User updated successfully');

            return new WP_REST_Response($this->prepare_user_data($updated_user), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Exception in update_user: ' . $e->getMessage());
            return new WP_Error(
                'update_user_error',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Delete user
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_user($request) {
        require_once(ABSPATH . 'wp-admin/includes/user.php');

        $user_id = (int) $request->get_param('id');
        $user = get_userdata($user_id);

        if (!$user) {
            return new WP_Error(
                'user_not_found',
                __('User not found', 'nexawp-connector'),
                array('status' => 404)
            );
        }

        // Don't allow deletion of the only admin
        if ($this->is_last_admin($user_id)) {
            return new WP_Error(
                'last_admin',
                __('Cannot delete the last administrator', 'nexawp-connector'),
                array('status' => 403)
            );
        }

        $reassign_user = $request->get_param('reassign') ?: null;
        $deleted = wp_delete_user($user_id, $reassign_user);

        if (!$deleted) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete user', 'nexawp-connector'),
                array('status' => 500)
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('User deleted successfully', 'nexawp-connector'),
            'deleted' => true
        ), 200);
    }

    /**
     * Prepare user data for response
     *
     * @param WP_User $user User object
     * @return array
     */
    private function prepare_user_data($user) {
        // Get last login time if available
        $last_login = get_user_meta($user->ID, 'last_login', true);

        // Check if user is the last admin
        $is_last_admin = in_array('administrator', $user->roles) && $this->is_last_admin($user->ID);

        return array(
            'id'           => $user->ID,
            'username'     => $user->user_login,
            'email'        => $user->user_email,
            'display_name' => $user->display_name,
            'first_name'   => $user->first_name,
            'last_name'    => $user->last_name,
            'website'      => $user->user_url,
            'roles'        => $user->roles,
            'registered'   => mysql_to_rfc3339($user->user_registered),
            'capabilities' => array_keys($user->allcaps),
            'avatar_url'   => get_avatar_url($user->ID),
            'last_login'   => $last_login ? mysql_to_rfc3339($last_login) : null,
            'is_last_admin' => $is_last_admin
        );
    }

    /**
     * Check if user is the last administrator
     *
     * @param int $user_id User ID
     * @return bool
     */
    private function is_last_admin($user_id) {
        $user = get_userdata($user_id);
        if (!in_array('administrator', $user->roles)) {
            return false;
        }

        $admin_count = count(get_users(array(
            'role'    => 'administrator',
            'exclude' => array($user_id),
            'fields'  => 'ID',
        )));

        return $admin_count === 0;
    }

    /**
     * Validate password strength
     *
     * @param string $password The password to check
     * @return bool True if password is strong enough
     */
    private function is_password_strong($password) {
        // Password should be at least 8 characters
        if (strlen($password) < 8) {
            return false;
        }

        $strength = 0;

        // Contains lowercase letters
        if (preg_match('/[a-z]/', $password)) {
            $strength++;
        }

        // Contains uppercase letters
        if (preg_match('/[A-Z]/', $password)) {
            $strength++;
        }

        // Contains numbers
        if (preg_match('/[0-9]/', $password)) {
            $strength++;
        }

        // Contains special characters
        if (preg_match('/[^a-zA-Z0-9]/', $password)) {
            $strength++;
        }

        // Strong password should meet at least 3 of 4 criteria
        return $strength >= 4;
    }

    /**
     * Bulk delete users
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_delete_users($request) {
        require_once(ABSPATH . 'wp-admin/includes/user.php');

        $user_ids = $request->get_param('user_ids');

        if (!is_array($user_ids) || empty($user_ids)) {
            return new WP_Error(
                'invalid_user_ids',
                __('Invalid or empty user IDs array', 'nexawp-connector'),
                array('status' => 400)
            );
        }

        $results = array();

        foreach ($user_ids as $user_id) {
            $user_id = (int) $user_id;
            $user = get_userdata($user_id);

            if (!$user) {
                $results[$user_id] = array(
                    'success' => false,
                    'message' => __('User not found', 'nexawp-connector')
                );
                continue;
            }

            // Don't allow deletion of the only admin
            if ($this->is_last_admin($user_id)) {
                $results[$user_id] = array(
                    'success' => false,
                    'message' => __('Cannot delete the last administrator', 'nexawp-connector')
                );
                continue;
            }

            // Log what we're about to do
            error_log(sprintf('[NexaWP] Bulk deleting user ID %d (%s)', $user_id, $user->user_login));

            $reassign_user = $request->get_param('reassign') ?: null;
            $deleted = wp_delete_user($user_id, $reassign_user);

            if (!$deleted) {
                $results[$user_id] = array(
                    'success' => false,
                    'message' => __('Failed to delete user', 'nexawp-connector')
                );
                continue;
            }

            $results[$user_id] = array(
                'success' => true,
                'message' => __('User deleted successfully', 'nexawp-connector'),
                'username' => $user->user_login
            );
        }

        // Count successes and failures
        $success_count = count(array_filter($results, function($result) {
            return isset($result['success']) && $result['success'] === true;
        }));

        $error_count = count($results) - $success_count;

        return new WP_REST_Response(array(
            'success' => true,
            'message' => sprintf(
                __('%d users deleted successfully, %d failed', 'nexawp-connector'),
                $success_count,
                $error_count
            ),
            'results' => $results,
            'success_count' => $success_count,
            'error_count' => $error_count
        ), 200);
    }

    /**
     * Get bulk delete user arguments
     *
     * @return array
     */
    private function get_bulk_delete_user_args() {
        return array(
            'user_ids' => array(
                'required'          => true,
                'type'             => 'array',
                'sanitize_callback' => function($ids) {
                    return array_map('absint', $ids);
                },
            ),
            'reassign' => array(
                'type'             => 'integer',
                'sanitize_callback' => 'absint',
            ),
        );
    }

    /**
     * Arguments for get users endpoint
     *
     * @return array
     */
    private function get_users_args() {
        return array(
            'per_page' => array(
                'default'           => 100,
                'type'             => 'integer',
                'sanitize_callback' => 'absint',
            ),
            'page' => array(
                'default'           => 1,
                'type'             => 'integer',
                'sanitize_callback' => 'absint',
            ),
            'role' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'search' => array(
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'orderby' => array(
                'default'           => 'login',
                'type'             => 'string',
                'enum'             => array('ID', 'login', 'email', 'registered'),
            ),
            'order' => array(
                'default'           => 'ASC',
                'type'             => 'string',
                'enum'             => array('ASC', 'DESC'),
            ),
        );
    }

    /**
     * Verify API key
     *
     * @param WP_REST_Request $request Request object
     * @return bool
     */
    public function verify_api_key($request) {
        $api_key_request = $request->get_header('Authorization');
        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');
        return $api_key_request === $api_key_stored;
    }
}
