<?php
/**
 * NexaWP Themes Controller
 *
 * Handles all theme-related REST API endpoints
 *
 * @package NexaWP
 */

class NexaWP_Themes_Controller {

    /**
     * API Namespace
     *
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Register REST routes for themes
     */
    public function register_routes() {
        error_log('[NexaWP] Registering theme routes');

        // Get all themes
        register_rest_route($this->namespace, '/themes', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_themes'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Update theme status
        register_rest_route($this->namespace, '/themes/status', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'update_theme_status'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_theme_status_args(),
        ));

        // Update theme
        register_rest_route($this->namespace, '/themes/update', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'update_theme'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => $this->get_theme_update_args(),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        // Delete theme
        register_rest_route($this->namespace, '/themes/delete', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'delete_theme'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => $this->get_theme_delete_args(),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        error_log('[NexaWP] Theme routes registered');
    }

    /**
     * Get all themes
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_themes($request) {
        error_log('[NexaWP] Starting get_themes request');

        try {
            require_once ABSPATH . 'wp-admin/includes/theme.php';

            $all_themes = wp_get_themes();
            $current_theme = wp_get_theme();
            $themes_data = array();

            error_log('[NexaWP] Found ' . count($all_themes) . ' themes');

            foreach ($all_themes as $theme_slug => $theme) {
                $has_update = false;
                $update_themes = get_site_transient('update_themes');

                if (isset($update_themes->response[$theme_slug])) {
                    $has_update = true;
                }

                $screenshot = $theme->get_screenshot() ? get_theme_root_uri() . "/{$theme_slug}/" . $theme->get_screenshot() : '';
                error_log("[NexaWP] Processing theme: {$theme_slug}, Screenshot: {$screenshot}");

                $themes_data[] = array(
                    'name'        => sanitize_text_field($theme->get('Name')),
                    'slug'        => $theme_slug,
                    'version'     => sanitize_text_field($theme->get('Version')),
                    'author'      => wp_strip_all_tags($theme->get('Author')),
                    'description' => wp_strip_all_tags($theme->get('Description')),
                    'screenshot'  => $screenshot,
                    'is_active'   => ($current_theme->get_stylesheet() === $theme_slug),
                    'has_update'  => $has_update,
                    'update_details' => isset($update_themes->response[$theme_slug]) ? $update_themes->response[$theme_slug] : null,
                );
            }

            error_log('[NexaWP] Successfully prepared themes response');
            return new WP_REST_Response($themes_data, 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in get_themes: ' . $e->getMessage());
            return new WP_Error(
                'themes_error',
                __('Failed to retrieve themes', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Update theme status (activate)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_theme_status($request) {
        error_log('[NexaWP] Starting update_theme_status request');

        try {
            require_once ABSPATH . 'wp-admin/includes/theme.php';

            $theme_slug = sanitize_text_field($request->get_param('theme_slug'));
            error_log("[NexaWP] Attempting to activate theme: {$theme_slug}");

            // Verify theme exists
            $theme = wp_get_theme($theme_slug);
            if (!$theme->exists()) {
                error_log("[NexaWP] Theme not found: {$theme_slug}");
                return new WP_Error(
                    'theme_not_found',
                    __('Theme not found', 'nexawp-connector'),
                    array('status' => 404)
                );
            }

            // Switch theme
            switch_theme($theme_slug);
            error_log("[NexaWP] Theme activated successfully: {$theme_slug}");

            return new WP_REST_Response(array(
                'success' => true,
                'message' => __('Theme activated successfully', 'nexawp-connector'),
                'slug'    => $theme_slug
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in update_theme_status: ' . $e->getMessage());
            return new WP_Error(
                'theme_activation_error',
                __('Failed to activate theme', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Update theme
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_theme($request) {
        error_log('[NexaWP] Starting theme update request');

        try {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/theme.php';
            require_once ABSPATH . 'wp-admin/includes/class-theme-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/misc.php';

            $theme_slug = sanitize_text_field($request->get_param('theme_slug'));
            error_log("[NexaWP] ========= STARTING THEME UPDATE PROCESS =========");
            error_log("[NexaWP] Request headers: " . json_encode(getallheaders()));
            error_log("[NexaWP] Theme to update: {$theme_slug}");

            // Initialize WP_Filesystem
            WP_Filesystem();
            error_log("[NexaWP] WP_Filesystem initialized");

            // Get theme information before update
            $theme = wp_get_theme($theme_slug);
            $old_version = $theme->get('Version');
            $is_active = ($theme->get_stylesheet() === $theme_slug);

            error_log("[NexaWP] Initial theme state - Active: " . ($is_active ? 'yes' : 'no') . ', Version: ' . $old_version);
            error_log("[NexaWP] Memory usage before update: " . memory_get_usage(true));

            // Check if theme has an update available
            $update_themes = get_site_transient('update_themes');
            if (!isset($update_themes->response[$theme_slug])) {
                error_log("[NexaWP] No update available for theme: {$theme_slug}");
                return new WP_Error(
                    'no_update_available',
                    __('No update available for this theme', 'nexawp-connector'),
                    array('status' => 400)
                );
            }

            $new_version = $update_themes->response[$theme_slug]['new_version'] ?? '';
            error_log("[NexaWP] New version available: {$new_version}");

            // Perform the update
            $skin = new Automatic_Upgrader_Skin();
            $upgrader = new Theme_Upgrader($skin);
            $result = $upgrader->upgrade($theme_slug);

            if (is_wp_error($result)) {
                error_log("[NexaWP] Theme update failed: " . $result->get_error_message());
                return $result;
            }

            if (false === $result) {
                error_log("[NexaWP] Theme update failed with unknown error");
                return new WP_Error(
                    'update_failed',
                    __('Theme update failed', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            // Verify the update
            $updated_theme = wp_get_theme($theme_slug);
            $actual_new_version = $updated_theme->get('Version');
            error_log("[NexaWP] New version installed: {$actual_new_version}");

            if (version_compare($actual_new_version, $old_version, '>')) {
                error_log("[NexaWP] Update verification successful");

                return new WP_REST_Response(array(
                    'success' => true,
                    'data' => array(
                        'slug' => $theme_slug,
                        'name' => $updated_theme->get('Name'),
                        'old_version' => $old_version,
                        'version' => $actual_new_version,
                        'status' => 'completed',
                        'is_active' => $is_active,
                        'updated_at' => current_time('mysql')
                    ),
                    'message' => __('Theme update completed successfully', 'nexawp-connector')
                ), 200);
            } else {
                error_log("[NexaWP] Update verification failed. Old: {$old_version} New: {$actual_new_version}");
                return new WP_Error(
                    'update_verification_failed',
                    __('Theme update could not be verified', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

        } catch (Exception $e) {
            error_log('[NexaWP] Critical error during update: ' . $e->getMessage());
            error_log('[NexaWP] Error trace: ' . $e->getTraceAsString());
            return new WP_Error(
                'update_error',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Delete a theme
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_theme($request) {
        error_log('[NexaWP] Starting theme deletion request');

        try {
            require_once ABSPATH . 'wp-admin/includes/theme.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';

            $theme_slug = sanitize_text_field($request->get_param('theme_slug'));
            error_log("[NexaWP] Attempting to delete theme: {$theme_slug}");

            $theme = wp_get_theme($theme_slug);
            if (!$theme->exists()) {
                error_log("[NexaWP] Theme not found: {$theme_slug}");
                return new WP_Error(
                    'theme_not_found',
                    __('Theme not found', 'nexawp-connector'),
                    array('status' => 404)
                );
            }

            $current_theme = wp_get_theme();
            if ($current_theme->get_stylesheet() === $theme_slug) {
                error_log("[NexaWP] Cannot delete active theme: {$theme_slug}");
                return new WP_Error(
                    'active_theme',
                    __('Cannot delete the active theme', 'nexawp-connector'),
                    array('status' => 403)
                );
            }

            $default_themes = array(
                'twentytwentyfive', 'twentytwentyfour', 'twentytwentythree',
                'twentytwentytwo', 'twentytwentyone', 'twentytwenty',
                'twentynineteen', 'twentyeighteen', 'twentyseventeen'
            );

            $all_themes = wp_get_themes();
            $non_default_themes = array_filter($all_themes, function($t) use ($default_themes) {
                return !in_array($t->get_stylesheet(), $default_themes);
            });

            if (in_array($theme_slug, $default_themes) && (count($all_themes) <= 2 || count($non_default_themes) == 0)) {
                error_log("[NexaWP] Cannot delete default theme when it's one of the last available: {$theme_slug}");
                return new WP_Error(
                    'default_theme_protection',
                    __('Cannot delete a default WordPress theme when there are no other themes available', 'nexawp-connector'),
                    array('status' => 403)
                );
            }

            $wp_filesystem = new WP_Filesystem_Direct(array());

            $theme_dir = $theme->get_stylesheet_directory();

            if (!$wp_filesystem->is_writable(dirname($theme_dir))) {
                error_log("[NexaWP] Theme directory is not writable: {$theme_dir}");
                return new WP_Error(
                    'filesystem_error',
                    __('Unable to delete theme: directory is not writable', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            $deleted = $wp_filesystem->delete($theme_dir, true);

            if (!$deleted) {
                error_log("[NexaWP] Failed to delete theme directory: {$theme_dir}");
                return new WP_Error(
                    'deletion_failed',
                    __('Failed to delete theme files', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            wp_cache_delete('themes', 'themes');

            error_log("[NexaWP] Theme deleted successfully: {$theme_slug}");

            return new WP_REST_Response(array(
                'success' => true,
                'message' => sprintf(__('Theme "%s" deleted successfully', 'nexawp-connector'), $theme->get('Name')),
                'data' => array(
                    'slug' => $theme_slug,
                    'name' => $theme->get('Name')
                )
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in delete_theme: ' . $e->getMessage());
            return new WP_Error(
                'theme_deletion_error',
                __('Failed to delete theme: ', 'nexawp-connector') . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Arguments for theme deletion
     *
     * @return array
     */
    private function get_theme_delete_args() {
        return array(
            'theme_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_theme_exists'),
            ),
        );
    }

    /**
     * Arguments for theme status update
     *
     * @return array
     */
    private function get_theme_status_args() {
        return array(
            'theme_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_theme_exists'),
            ),
        );
    }

    /**
     * Arguments for theme update
     *
     * @return array
     */
    private function get_theme_update_args() {
        return array(
            'theme_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_theme_exists'),
            ),
        );
    }

    /**
     * Validate theme exists
     *
     * @param string $slug Theme slug
     * @return bool
     */
    public function validate_theme_exists($slug) {
        $exists = wp_get_theme($slug)->exists();
        error_log("[NexaWP] Validating theme exists: {$slug} - " . ($exists ? 'true' : 'false'));
        return $exists;
    }

    /**
     * Verify API key
     *
     * @param WP_REST_Request $request Request object
     * @return bool
     */
    public function verify_api_key($request) {
        $api_key_request = $request->get_header('Authorization');
        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');

        error_log('[NexaWP] Verifying API key - Request: ' . substr($api_key_request, -4) . ' Stored: ' . substr($api_key_stored, -4));

        return $api_key_request === $api_key_stored;
    }
}
