<?php
/**
 * NexaWP REST API Main Controller
 *
 * Main controller that initializes and coordinates all other API controllers
 *
 * @package NexaWP
 */

class NexaWP_REST_Controller {
    /**
     * API Namespace
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Controller instances
     * @var array
     */
    private $controllers = array();

    /**
     * Allowed origin for CORS
     * @var string
     */
    private $allowed_origin = 'https://app.nexawpmanager.com';

    /**
     * Constructor
     */
    public function __construct() {
        add_action('init', array($this, 'add_cors_headers'), 1);
        add_action('rest_api_init', array($this, 'init_api'), 10);
    }

    /**
     * Initialize plugin
     */
    public function init_plugin() {
        // Check if permalinks are enabled
        if (!get_option('permalink_structure')) {
            add_action('admin_notices', function() {
                echo '<div class="error"><p>';
                echo __('NexaWP Connector requires WordPress Permalinks to be enabled. Please enable them in Settings > Permalinks.', 'nexawp-connector');
                echo '</p></div>';
            });
        }

        flush_rewrite_rules();

        $this->load_controllers();
        $this->init_hooks();
    }

    /**
     * Initialize API with security plugin compatibility
     */
    public function init_api() {
        error_log('[NexaWP] Initializing API');

        // Temporarily disable common security restrictions for our endpoints
        add_filter('secupress.plugin.rest_api', '__return_true', 999);
        add_filter('secupress.plugin.api_access', '__return_true', 999);

        // For other security plugins
        if (isset($_SERVER['HTTP_X_CUSTOM_PLUGIN']) && $_SERVER['HTTP_X_CUSTOM_PLUGIN'] === 'NexaWP') {
            add_filter('rest_authentication_errors', '__return_true', 999);
        }

        if (!$this->check_rest_api_status()) {
            error_log('[NexaWP] REST API check failed');
            return;
        }

        $this->init_controllers();
        $this->register_routes();

        remove_filter('rest_pre_serve_request', 'rest_send_cors_headers');
        add_filter('rest_pre_serve_request', array($this, 'handle_cors_headers'));

        error_log('[NexaWP] API initialized');
    }

    /**
     * Initialize all controllers
     */
    public function init_controllers() {
        error_log('[NexaWP] Initializing controllers');

        $this->controllers = array(
            'plugins' => new NexaWP_Plugins_Controller(),
            'themes'  => new NexaWP_Themes_Controller(),
            'users'   => new NexaWP_Users_Controller(),
            'backup'  => new NexaWP_Backup_Controller(),
            'core'    => new NexaWP_Core_Controller(),
        );
    }

    /**
     * Register all REST routes
     */
    public function register_routes() {
        // Register routes in a specific order
        if (isset($this->controllers['users'])) {
            $this->controllers['users']->register_routes();
            error_log('[NexaWP] Users routes registered');
        }

        if (isset($this->controllers['plugins'])) {
            $this->controllers['plugins']->register_routes();
            error_log('[NexaWP] Plugin routes registered');
        }

        if (isset($this->controllers['themes'])) {
            $this->controllers['themes']->register_routes();
            error_log('[NexaWP] Theme routes registered');
        }

        if (isset($this->controllers['core'])) {
            $this->controllers['core']->register_routes();
            error_log('[NexaWP] Core routes registered');
        }

        if (isset($this->controllers['backup'])) {
            $this->controllers['backup']->register_routes();
            error_log('[NexaWP] Backup routes registered');
        }

        // Authenticated routes
        register_rest_route($this->namespace, '/verify', array(
            'methods'             => array('GET', 'OPTIONS'),
            'callback'            => array($this, 'verify_connection'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        register_rest_route($this->namespace, '/site/info', array(
            'methods'             => array('GET', 'OPTIONS'),
            'callback'            => array($this, 'get_site_info'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        register_rest_route($this->namespace, '/auto-login', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'handle_auto_login' ],
            'permission_callback' => [ $this, 'verify_api_key' ],
        ]);

        // Log registered routes
        global $wp_rest_server;
        if ($wp_rest_server) {
            $routes = $wp_rest_server->get_routes();
            foreach ($routes as $route => $handlers) {
                if (strpos($route, 'nexawp') !== false) {
                    error_log("[NexaWP] Registered route: {$route}");
                }
            }
        }

        error_log('[NexaWP] Routes registered successfully');
    }

    /**
     * Handle the auto-login logic
     */
    public function handle_auto_login($request) {
        // 1) Already logged in?
        if ( is_user_logged_in() ) {
            return new WP_REST_Response([
                'success' => true,
                'redirect_url' => admin_url(),
            ], 200);
        }

        $token = wp_generate_password(32, false);

        $admin_user = get_users([
            'role'   => 'administrator',
            'number' => 1,
        ]);
        if (empty($admin_user)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'No admin user found',
            ], 500);
        }
        $admin_user_id = $admin_user[0]->ID;

        set_transient("nexawp_auto_login_$token", $admin_user_id, 60);

        $magic_url = add_query_arg('nexawp_auto_login_token', $token, home_url('/'));

        return new WP_REST_Response([
            'success' => true,
            'redirect_url' => $magic_url,
        ], 200);
    }

    /**
     * Check REST API status
     *
     * @return bool
     */
    private function check_rest_api_status() {
        if (!get_option('permalink_structure')) {
            error_log('[NexaWP] Permalinks are not enabled');
            return false;
        }

        if (defined('REST_REQUEST') && REST_REQUEST) {
            error_log('[NexaWP] REST API is active');
            return true;
        }

        error_log('[NexaWP] REST API might be disabled');
        return false;
    }

    /**
     * Get allowed origin for CORS
     *
     * @return string
     */
    private function get_allowed_origin() {
        return defined('NEXAWP_MANAGER_URL') ? NEXAWP_MANAGER_URL : 'https://app.nexawpmanager.com';
    }

    /**
     * Add CORS headers with enhanced security plugin compatibility
     */
    public function add_cors_headers() {
        $origin = isset($_SERVER['HTTP_ORIGIN']) ? $_SERVER['HTTP_ORIGIN'] : '';
        $allowed_origin = $this->get_allowed_origin();

        error_log('[NexaWP] Request Origin: ' . $origin);
        error_log('[NexaWP] Allowed Origin: ' . $allowed_origin);

        // Add standard CORS headers - INCLUDE X-App-Locale
        header('Access-Control-Allow-Origin: ' . $allowed_origin);
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS, PATCH');
        header('Access-Control-Allow-Headers: Authorization, Content-Type, Accept, Origin, X-Requested-With, X-WP-Nonce, X-Custom-Plugin, X-App-Locale, X-App-Token, X-App-Version');
        header('Access-Control-Allow-Credentials: false');
        header('Access-Control-Max-Age: 86400');
        header('Vary: Origin');

        // Add security compatibility headers
        $this->add_security_compatibility_headers();

        // Handle preflight request
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            status_header(200);
            exit();
        }
    }

    /**
     * Add custom headers for security plugin compatibility
     * This ensures our requests can pass through common security plugins
     */
    private function add_security_compatibility_headers() {
        // Add custom header for SecuPress compatibility
        if (!headers_sent()) {
            header('X-WP-Nonce: ' . wp_create_nonce('wp_rest'));

            // Add headers that security plugins typically check
            header('X-Requested-With: XMLHttpRequest');
            header('X-Custom-Plugin: NexaWP');

            // Set a more standard origin header
            if (isset($_SERVER['HTTP_ORIGIN'])) {
                header('Origin: ' . $_SERVER['HTTP_ORIGIN']);
            }
        }
    }

    /**
     * Handle CORS headers for requests
     *
     * @param bool $served Whether the request has already been served
     * @return bool
     */
    public function handle_cors_headers($served) {
        $origin = isset($_SERVER['HTTP_ORIGIN']) ? $_SERVER['HTTP_ORIGIN'] : '';
        $allowed_origin = $this->get_allowed_origin();

        if ($origin === $allowed_origin) {
            header('Access-Control-Allow-Origin: ' . $allowed_origin);
            header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS, PATCH');
            header('Access-Control-Allow-Headers: Authorization, Content-Type, Accept, Origin, X-Requested-With, X-WP-Nonce, X-App-Locale, X-App-Token, X-App-Version');
            header('Access-Control-Allow-Credentials: false');
            header('Access-Control-Max-Age: 86400');
            header('Vary: Origin');
        }

        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            status_header(200);
            exit();
        }

        return $served;
    }

    /**
     * Verify connection
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response
     */
    public function verify_connection($request) {
        error_log('[NexaWP] Verifying connection');
        if (isset($_SERVER['HTTP_ORIGIN'])) {
            error_log('[NexaWP] Connection from origin: ' . $_SERVER['HTTP_ORIGIN']);
        }

        return new WP_REST_Response(array(
            'status'    => 'connected',
            'message'   => 'Successfully connected to the site',
            'version'   => NEXAWP_VERSION,
            'timestamp' => current_time('mysql'),
        ), 200);
    }

    /**
     * Get site information
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response
     */
    public function get_site_info($request) {
        $info = array(
            'name'                => get_bloginfo('name'),
            'description'         => get_bloginfo('description'),
            'url'                 => get_site_url(),
            'admin_url'           => admin_url(),
            'version'             => get_bloginfo('version'),
            'language'            => get_locale(),
            'timezone'            => get_option('timezone_string'),
            'date_format'         => get_option('date_format'),
            'time_format'         => get_option('time_format'),
            'site_icon'           => get_site_icon_url(),
            'favicon'             => $this->get_favicon_url(),  // Added favicon field
            'permalink_structure' => get_option('permalink_structure'),
            'is_multisite'        => is_multisite(),
            'api_version'         => NEXAWP_VERSION,
            'system_info'         => $this->get_system_info(),
        );

        return new WP_REST_Response($info, 200);
    }

    /**
     * Get the favicon URL of the site
     *
     * @return string
     */
    private function get_favicon_url() {
        // 1. Prioritize WordPress site icon (the proper way)
        $site_icon_id = get_option('site_icon');
        if ($site_icon_id) {
            $icon_url = wp_get_attachment_image_url($site_icon_id, 'full');
            if ($icon_url) {
                return $icon_url;
            }
        }

        // 2. Try get_site_icon_url() as fallback
        $site_icon_url = get_site_icon_url();
        if (!empty($site_icon_url)) {
            return $site_icon_url;
        }

        // 3. Look for favicon.ico in the root directory
        $favicon_path = ABSPATH . 'favicon.ico';
        if (file_exists($favicon_path)) {
            return home_url('/favicon.ico');
        }

        // 4. Check for common favicon locations in theme
        $theme_dir = get_template_directory();
        $theme_url = get_template_directory_uri();

        $favicon_locations = [
            '/favicon.ico',
            '/favicon.png',
            '/favicon.jpg',
            '/images/favicon.ico',
            '/images/favicon.png',
            '/assets/images/favicon.ico',
            '/assets/images/favicon.png',
            '/assets/favicon.ico',
            '/assets/favicon.png',
        ];

        foreach ($favicon_locations as $location) {
            if (file_exists($theme_dir . $location)) {
                return $theme_url . $location;
            }
        }

        // 5. Check for link tags in the HTML header
        // This is more complex and would require parsing the site's HTML

        // Return empty if no favicon found
        return '';
    }

    /**
     * Get system information
     *
     * @return array
     */
    private function get_system_info() {
        global $wpdb;

        return array(
            'php_version'            => phpversion(),
            'mysql_version'          => $wpdb->db_version(),
            'web_server'             => isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field($_SERVER['SERVER_SOFTWARE']) : '',
            'wordpress_memory_limit' => WP_MEMORY_LIMIT,
            'php_memory_limit'       => ini_get('memory_limit'),
            'debug_mode'             => defined('WP_DEBUG') && WP_DEBUG,
            'max_upload_size'        => size_format(wp_max_upload_size()),
            'php_post_max_size'      => ini_get('post_max_size'),
            'curl_version'           => function_exists('curl_version') ? curl_version()['version'] : 'N/A',
            'ssl_installed'          => is_ssl(),
        );
    }

    /**
     * Verify API key with enhanced security plugin compatibility
     *
     * @param WP_REST_Request $request The request object
     * @return bool|WP_Error
     */
    public function verify_api_key($request) {
        error_log('[NexaWP] Starting API key verification');

        // Try multiple authorization header variations
        $auth_headers = [
            $request->get_header('Authorization'),
            $request->get_header('HTTP_AUTHORIZATION'),
            isset($_SERVER['HTTP_AUTHORIZATION']) ? $_SERVER['HTTP_AUTHORIZATION'] : null,
            isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION']) ? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] : null
        ];

        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');

        foreach ($auth_headers as $api_key_request) {
            if ($api_key_request && $api_key_request === $api_key_stored) {
                error_log('[NexaWP] API key verification successful');
                return true;
            }
        }

        error_log('[NexaWP] Invalid API key comparison');
        return new WP_Error(
            'auth_failed',
            'Invalid API key',
            array('status' => 403)
        );
    }
}
