<?php
/**
 * NexaWP Plugins Controller
 *
 * Handles all plugin-related REST API endpoints
 *
 * @package NexaWP
 */

class NexaWP_Plugins_Controller {

    /**
     * API Namespace
     *
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Register REST routes for plugins
     */
    public function register_routes() {
        error_log('[NexaWP] Registering plugin routes');

        // Get all plugins
        register_rest_route($this->namespace, '/plugins', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_plugins'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Update plugin status
        register_rest_route($this->namespace, '/plugins/status', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'update_plugin_status'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_plugin_status_args(),
        ));

        // Update plugin
        register_rest_route($this->namespace, '/plugins/update', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'update_plugin'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_plugin_update_args(),
        ));

        // Bulk update plugins
        register_rest_route($this->namespace, '/plugins/bulk-update', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'bulk_update_plugins'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_bulk_update_args(),
        ));

        // Auto-update
        register_rest_route($this->namespace, '/plugins/auto-update', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'toggle_auto_update'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_auto_update_args(),
        ));

        // Rollback
        register_rest_route($this->namespace, '/plugins/rollback', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'rollback_plugin'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => $this->get_rollback_args(),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        // Delete plugin
        register_rest_route($this->namespace, '/plugins/delete', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'delete_plugin'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => $this->get_plugin_delete_args(),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        error_log('[NexaWP] Plugin routes registered');
    }

    /**
     * Get all plugins
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_plugins($request) {
        error_log('[NexaWP] Starting get_plugins request');

        try {
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            $all_plugins = get_plugins();
            if (!is_array($all_plugins)) {
                $all_plugins = array();
            }

            $active_plugins = get_option('active_plugins', array());
            $plugins_data = array();

            error_log('[NexaWP] Found ' . count($all_plugins) . ' plugins');

            foreach ($all_plugins as $plugin_file => $plugin_data) {
                $has_update = false;
                $update_plugins = get_site_transient('update_plugins');
                $auto_update = false;
                $screenshot_url = '';

                // Check for updates
                if (isset($update_plugins->response[$plugin_file])) {
                    $has_update = true;
                }

                // Check auto-update status
                if (function_exists('wp_is_auto_update_enabled_for_type') &&
                    wp_is_auto_update_enabled_for_type('plugin')) {
                    $auto_update = in_array($plugin_file, (array) get_site_option('auto_update_plugins', array()));
                }

                // Get plugin slug for WordPress.org API
                $plugin_slug = dirname($plugin_file);
                if ($plugin_slug === '.') {
                    $plugin_slug = basename($plugin_file, '.php');
                }

                // Get plugin info from WordPress.org API
                if (!empty($plugin_slug)) {
                    $api_url = "https://api.wordpress.org/plugins/info/1.2/?action=plugin_information&slug=" . $plugin_slug;
                    $response = wp_remote_get($api_url);

                    if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                        $plugin_info = json_decode(wp_remote_retrieve_body($response));

                        // Try to get icon from plugin info
                        if ($plugin_info && !empty($plugin_info->icons)) {
                            // Prefer high-resolution icons
                            if (!empty($plugin_info->icons['2x'])) {
                                $screenshot_url = $plugin_info->icons['2x'];
                            } elseif (!empty($plugin_info->icons['1x'])) {
                                $screenshot_url = $plugin_info->icons['1x'];
                            } elseif (!empty($plugin_info->icons['svg'])) {
                                $screenshot_url = $plugin_info->icons['svg'];
                            } elseif (!empty($plugin_info->icons['default'])) {
                                $screenshot_url = $plugin_info->icons['default'];
                            }
                        }

                        // If no icon found, try to get from wp.org assets directly
                        if (empty($screenshot_url)) {
                            $wp_org_icon = "https://ps.w.org/{$plugin_slug}/assets/icon-256x256.";
                            $extensions = ['jpg', 'jpeg', 'png'];

                            foreach ($extensions as $ext) {
                                $test_url = $wp_org_icon . $ext;
                                $response = wp_remote_head($test_url);
                                if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                                    $screenshot_url = $test_url;
                                    break;
                                }
                            }
                        }
                    }
                }

                // For plugins not on WordPress.org or without icons
                if (empty($screenshot_url)) {
                    // Try to get icon from update_details if available
                    if (isset($update_plugins->response[$plugin_file]) &&
                        isset($update_plugins->response[$plugin_file]->icons)) {
                        $icons = $update_plugins->response[$plugin_file]->icons;
                        if (!empty($icons['2x'])) {
                            $screenshot_url = $icons['2x'];
                        } elseif (!empty($icons['1x'])) {
                            $screenshot_url = $icons['1x'];
                        } elseif (!empty($icons['svg'])) {
                            $screenshot_url = $icons['svg'];
                        }
                    }
                }

                // Only use WordPress default if nothing else is found
                if (empty($screenshot_url)) {
                    $screenshot_url = includes_url('images/w-logo-blue.png');
                }

                $plugins_data[] = array(
                    'name'           => sanitize_text_field($plugin_data['Name']),
                    'slug'           => $plugin_file,
                    'version'        => sanitize_text_field($plugin_data['Version']),
                    'author'         => wp_strip_all_tags($plugin_data['Author']),
                    'description'    => wp_strip_all_tags($plugin_data['Description']),
                    'is_active'      => in_array($plugin_file, $active_plugins),
                    'has_update'     => $has_update,
                    'auto_update'    => $auto_update,
                    'update_details' => isset($update_plugins->response[$plugin_file]) ? $update_plugins->response[$plugin_file] : null,
                    'screenshot'     => esc_url($screenshot_url),
                    'plugin_uri'     => isset($plugin_data['PluginURI']) ? esc_url_raw($plugin_data['PluginURI']) : '',
                    'author_uri'     => isset($plugin_data['AuthorURI']) ? esc_url_raw($plugin_data['AuthorURI']) : '',
                    'text_domain'    => isset($plugin_data['TextDomain']) ? sanitize_text_field($plugin_data['TextDomain']) : '',
                    'domain_path'    => isset($plugin_data['DomainPath']) ? sanitize_text_field($plugin_data['DomainPath']) : '',
                );
            }

            error_log('[NexaWP] Successfully prepared plugins response');
            return new WP_REST_Response($plugins_data, 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in get_plugins: ' . $e->getMessage());
            return new WP_Error(
                'plugin_error',
                __('Failed to retrieve plugins', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Update plugin status (activate/deactivate)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_plugin_status($request) {
        try {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';

            $plugin_slug = sanitize_text_field($request->get_param('plugin_slug'));
            $status = sanitize_text_field($request->get_param('status'));

            if ($plugin_slug === 'nexawp-connector/nexawp-connector.php') {
                return new WP_Error(
                    'nexawp_plugin_protected',
                    __('The NexaWP Connector plugin cannot be deactivated through the API', 'nexawp-connector'),
                    array('status' => 403)
                );
            }

            $plugin_info = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_slug);

            if ($status === 'active') {
                $result = activate_plugin($plugin_slug);
                $action = 'activated';
            } else {
                $result = deactivate_plugins($plugin_slug);
                $action = 'deactivated';
            }

            if (is_wp_error($result)) {
                return $result;
            }

            return new WP_REST_Response(array(
                'success' => true,
                'message' => sprintf(
                    __('Plugin %s successfully', 'nexawp-connector'),
                    $action
                ),
                'data' => array(
                    'slug' => $plugin_slug,
                    'name' => $plugin_info['Name'],
                    'status' => $status,
                    'version' => $plugin_info['Version'],
                    'is_active' => is_plugin_active($plugin_slug),
                    'timestamp' => current_time('mysql')
                )
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in update_plugin_status: ' . $e->getMessage());
            return new WP_Error(
                'status_update_error',
                __('Failed to update plugin status', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Auto-update
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */

    public function toggle_auto_update($request) {
        try {
            error_log('[NexaWP] Starting toggle_auto_update request');
            error_log('[NexaWP] Request parameters: ' . print_r($request->get_params(), true));

            $plugin_slug = sanitize_text_field($request->get_param('plugin_slug'));
            $auto_update = (bool) $request->get_param('auto_update');

            error_log('[NexaWP] Plugin slug: ' . $plugin_slug);
            error_log('[NexaWP] Auto update value: ' . ($auto_update ? 'true' : 'false'));

            $auto_update_plugins = (array) get_site_option('auto_update_plugins', array());
            error_log('[NexaWP] Current auto-update plugins: ' . print_r($auto_update_plugins, true));

            if ($auto_update) {
                $auto_update_plugins[] = $plugin_slug;
                $auto_update_plugins = array_unique($auto_update_plugins);
                error_log('[NexaWP] Added plugin to auto-update list');
            } else {
                $auto_update_plugins = array_diff($auto_update_plugins, array($plugin_slug));
                error_log('[NexaWP] Removed plugin from auto-update list');
            }

            $update_result = update_site_option('auto_update_plugins', $auto_update_plugins);
            error_log('[NexaWP] Update result: ' . ($update_result ? 'success' : 'failed'));

            return new WP_REST_Response(array(
                'success' => true,
                'message' => $auto_update ? 'Auto-update enabled' : 'Auto-update disabled',
                'data' => array(
                    'slug' => $plugin_slug,
                    'auto_update' => $auto_update,
                    'updated_at' => current_time('mysql')
                )
            ), 200);
        } catch (Exception $e) {
            error_log('[NexaWP] Error in toggle_auto_update: ' . $e->getMessage());
            error_log('[NexaWP] Stack trace: ' . $e->getTraceAsString());
            return new WP_Error(
                'auto_update_error',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Rollback
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */

    public function rollback_plugin($request) {
        try {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

            $plugin_slug = $request->get_param('plugin_slug');
            $version = $request->get_param('version');

            // Obtenir les informations du plugin
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_slug);
            $plugin_name = $plugin_data['Name'];

            // Construire l'URL de téléchargement de la version spécifique
            $download_url = sprintf(
                'https://downloads.wordpress.org/plugin/%s.%s.zip',
                dirname($plugin_slug),
                $version
            );

            // Sauvegarder l'état actif
            $was_active = is_plugin_active($plugin_slug);

            // Désactiver temporairement
            if ($was_active) {
                deactivate_plugins($plugin_slug);
            }

            // Effectuer le rollback
            $upgrader = new Plugin_Upgrader(new WP_Ajax_Upgrader_Skin());
            $result = $upgrader->install($download_url, array('overwrite_package' => true));

            // Réactiver si nécessaire
            if ($was_active) {
                activate_plugin($plugin_slug);
            }

            if (is_wp_error($result)) {
                throw new Exception($result->get_error_message());
            }

            return new WP_REST_Response(array(
                'success' => true,
                'message' => sprintf(__('Successfully rolled back %s to version %s', 'nexawp-connector'), $plugin_name, $version),
                'data' => array(
                    'slug' => $plugin_slug,
                    'version' => $version,
                    'is_active' => $was_active
                )
            ));

        } catch (Exception $e) {
            return new WP_Error(
                'rollback_failed',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Get rollback arguments
     *
     * @return array
     */
    private function get_rollback_args() {
        return array(
            'plugin_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_plugin_exists'),
            ),
            'version' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ),
        );
    }

    private function get_auto_update_args() {
        return array(
            'plugin_slug' => array(
                'required' => true,
                'type'     => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_plugin_exists'),
            ),
            'auto_update' => array(
                'required' => true,
                'type'     => 'boolean',
            ),
        );
    }

    /**
     * Update a plugin via REST API with special handling for complex plugins
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_plugin($request) {
        try {
            error_log('[NexaWP] ========= STARTING PLUGIN UPDATE PROCESS =========');
            error_log('[NexaWP] Request headers: ' . json_encode(getallheaders()));

            // Required WordPress files
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/misc.php';
            include_once ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php';

            $plugin_slug = sanitize_text_field($request->get_param('plugin_slug'));
            error_log('[NexaWP] Plugin to update: ' . $plugin_slug);

            // Initialize WP_Filesystem
            WP_Filesystem();
            error_log('[NexaWP] WP_Filesystem initialized');

            // Store current state
            $was_active = is_plugin_active($plugin_slug);
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_slug);
            $old_version = $plugin_data['Version'];

            error_log('[NexaWP] Initial plugin state - Active: ' . ($was_active ? 'yes' : 'no') . ', Version: ' . $old_version);
            error_log('[NexaWP] Memory usage before update: ' . memory_get_usage(true));

            // Check for available updates
            wp_update_plugins();
            $update_plugins = get_site_transient('update_plugins');

            if (!isset($update_plugins->response[$plugin_slug])) {
                error_log('[NexaWP] No update available for: ' . $plugin_slug);
                return new WP_Error(
                    'no_update_available',
                    'No update available for this plugin',
                    array('status' => 400)
                );
            }

            $new_version = $update_plugins->response[$plugin_slug]->new_version;
            error_log('[NexaWP] New version available: ' . $new_version);

            // Special handling for WooCommerce
            if ($plugin_slug === 'woocommerce/woocommerce.php') {
                error_log('[NexaWP] WooCommerce detected, using special handling');

                if ($was_active) {
                    error_log('[NexaWP] Deactivating WooCommerce temporarily');
                    deactivate_plugins($plugin_slug);
                    wp_cache_flush();
                    sleep(2);
                }
            }

            // Perform the update
            error_log('[NexaWP] Starting upgrade process');
            $upgrader = new Plugin_Upgrader(new Automatic_Upgrader_Skin());
            $result = $upgrader->upgrade($plugin_slug);

            if (is_wp_error($result)) {
                error_log('[NexaWP] Update failed: ' . $result->get_error_message());
                return $result;
            }

            error_log('[NexaWP] Update process completed');

            // Verify the update
            $updated_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_slug);
            $actual_new_version = $updated_data['Version'];
            error_log('[NexaWP] New version installed: ' . $actual_new_version);

            // Handle reactivation
            if ($was_active) {
                error_log('[NexaWP] Attempting to reactivate plugin');

                if ($plugin_slug === 'woocommerce/woocommerce.php') {
                    // Multiple attempts for WooCommerce
                    for ($i = 0; $i < 3; $i++) {
                        error_log('[NexaWP] WooCommerce reactivation attempt ' . ($i + 1));
                        activate_plugin($plugin_slug);
                        wp_cache_flush();

                        if (is_plugin_active($plugin_slug)) {
                            error_log('[NexaWP] WooCommerce reactivation successful');
                            break;
                        }

                        error_log('[NexaWP] WooCommerce reactivation attempt failed, waiting before retry');
                        sleep(2);
                    }
                } else {
                    activate_plugin($plugin_slug);
                    wp_cache_flush();
                }
            }

            $is_active_now = is_plugin_active($plugin_slug);
            error_log('[NexaWP] Final plugin state - Active: ' . ($is_active_now ? 'yes' : 'no'));

            if (version_compare($actual_new_version, $old_version, '>')) {
                error_log('[NexaWP] Update verification successful');

                return new WP_REST_Response(array(
                    'success' => true,
                    'data' => array(
                        'slug' => $plugin_slug,
                        'name' => $updated_data['Name'],
                        'old_version' => $old_version,
                        'new_version' => $actual_new_version,
                        'status' => 'completed',
                        'is_active' => $is_active_now,
                        'updated_at' => current_time('mysql')
                    ),
                    'message' => 'Plugin update completed successfully'
                ), 200);
            } else {
                error_log('[NexaWP] Update verification failed. Old: ' . $old_version . ' New: ' . $actual_new_version);
                return new WP_Error(
                    'update_verification_failed',
                    'Plugin update could not be verified',
                    array('status' => 500)
                );
            }

        } catch (Exception $e) {
            error_log('[NexaWP] Critical error during update: ' . $e->getMessage());
            error_log('[NexaWP] Error trace: ' . $e->getTraceAsString());
            return new WP_Error(
                'update_error',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Bulk update plugins
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_update_plugins($request) {
        error_log('[NexaWP] Starting bulk plugin update request');

        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';

        try {
            $plugins = $request->get_param('plugins');
            if (!is_array($plugins)) {
                return new WP_Error(
                    'invalid_plugins_list',
                    __('Invalid plugins list', 'nexawp-connector'),
                    array('status' => 400)
                );
            }

            error_log('[NexaWP] Attempting to update ' . count($plugins) . ' plugins');

            $skin = new Automatic_Upgrader_Skin();
            $upgrader = new Plugin_Upgrader($skin);
            $results = array();

            foreach ($plugins as $plugin) {
                $result = $upgrader->upgrade($plugin);
                $results[$plugin] = is_wp_error($result) ? $result->get_error_message() : 'success';
                error_log('[NexaWP] Update result for ' . $plugin . ': ' . $results[$plugin]);
            }

            error_log('[NexaWP] Bulk plugin update completed');
            return new WP_REST_Response(array(
                'message' => __('Bulk update completed', 'nexawp-connector'),
                'results' => $results
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in bulk_update_plugins: ' . $e->getMessage());
            return new WP_Error(
                'bulk_update_error',
                __('Failed to perform bulk update', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Delete a plugin
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_plugin($request) {
        error_log('[NexaWP] Starting delete_plugin request');

        try {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';

            $plugin_slug = sanitize_text_field($request->get_param('plugin_slug'));
            error_log("[NexaWP] Attempting to delete plugin: {$plugin_slug}");

            // Check if plugin exists
            if (!file_exists(WP_PLUGIN_DIR . '/' . $plugin_slug)) {
                error_log("[NexaWP] Plugin file not found: {$plugin_slug}");
                return new WP_Error(
                    'plugin_not_found',
                    __('Plugin not found', 'nexawp-connector'),
                    array('status' => 404)
                );
            }

            // Protect NexaWP Connector from being deleted through its own API
            if ($plugin_slug === 'nexawp-connector/nexawp-connector.php') {
                error_log("[NexaWP] Cannot delete NexaWP Connector through its own API");
                return new WP_Error(
                    'nexawp_plugin_protected',
                    __('The NexaWP Connector plugin cannot be deleted through the API', 'nexawp-connector'),
                    array('status' => 403)
                );
            }

            // Check if plugin is active
            if (is_plugin_active($plugin_slug)) {
                error_log("[NexaWP] Cannot delete active plugin: {$plugin_slug}");
                return new WP_Error(
                    'active_plugin',
                    __('Cannot delete an active plugin. Please deactivate it first.', 'nexawp-connector'),
                    array('status' => 403)
                );
            }

            // Get plugin information before deletion
            $plugin_info = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_slug);
            $plugin_name = $plugin_info['Name'];

            // Initialize filesystem
            WP_Filesystem();
            $wp_filesystem = new WP_Filesystem_Direct(array());

            // Plugin directory path
            $plugin_dir = WP_PLUGIN_DIR . '/' . dirname($plugin_slug);
            $is_file = dirname($plugin_slug) === '.';

            if ($is_file) {
                // Single file plugin
                $deleted = $wp_filesystem->delete(WP_PLUGIN_DIR . '/' . $plugin_slug, false, 'f');
            } else {
                // Directory plugin
                $deleted = $wp_filesystem->delete($plugin_dir, true);
            }

            if (!$deleted) {
                error_log("[NexaWP] Failed to delete plugin: {$plugin_slug}");
                return new WP_Error(
                    'deletion_failed',
                    __('Failed to delete plugin files', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            // Clear plugin cache
            wp_cache_delete('plugins', 'plugins');

            error_log("[NexaWP] Plugin deleted successfully: {$plugin_slug}");

            return new WP_REST_Response(array(
                'success' => true,
                'message' => sprintf(__('Plugin "%s" deleted successfully', 'nexawp-connector'), $plugin_name),
                'data' => array(
                    'slug' => $plugin_slug,
                    'name' => $plugin_name
                )
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in delete_plugin: ' . $e->getMessage());
            return new WP_Error(
                'plugin_deletion_error',
                __('Failed to delete plugin: ', 'nexawp-connector') . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Bulk delete inactive plugins
     *
     * @param Request $request
     * @param WordPressSite $site
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkDeletePlugins(Request $request, WordPressSite $site)
    {
        try {
            $pluginSlugs = $request->input('plugin_slugs');

            Log::info('=== Starting bulk plugin deletion ===', [
                'site_id' => $site->id,
                'site_uuid' => $site->uuid,
                'site_url' => $site->url,
                'plugin_count' => count($pluginSlugs),
            ]);

            if (!is_array($pluginSlugs) || empty($pluginSlugs)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No plugins specified for deletion'
                ], 400);
            }

            $results = [];
            $successCount = 0;
            $failCount = 0;

            // Traiter chaque plugin séquentiellement
            foreach ($pluginSlugs as $pluginSlug) {
                try {
                    // Vérifier que le plugin existe dans la base de données
                    $plugin = $site->plugins()->where('slug', $pluginSlug)->first();

                    if (!$plugin) {
                        Log::warning('Plugin not found in database for bulk deletion', [
                            'plugin_slug' => $pluginSlug,
                            'site_id' => $site->id
                        ]);

                        $results[$pluginSlug] = [
                            'success' => false,
                            'message' => 'Plugin not found in database'
                        ];
                        $failCount++;
                        continue;
                    }

                    // Vérifier si le plugin est actif
                    if ($plugin->is_active) {
                        Log::warning('Cannot delete active plugin in bulk operation', [
                            'plugin_slug' => $pluginSlug,
                            'plugin_name' => $plugin->name
                        ]);

                        $results[$pluginSlug] = [
                            'success' => false,
                            'message' => 'Cannot delete active plugin'
                        ];
                        $failCount++;
                        continue;
                    }

                    // Appel à l'API WordPress pour supprimer le plugin
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $site->api_key,
                        'Accept' => 'application/json',
                    ])->post($site->url . '/wp-json/nexawp/v1/plugins/delete', [
                        'plugin_slug' => $pluginSlug
                    ]);

                    if (!$response->successful()) {
                        $errorMessage = $response->json()['message'] ?? 'Failed to delete plugin on WordPress site';

                        Log::error('WordPress API bulk delete request failed for plugin', [
                            'plugin_slug' => $pluginSlug,
                            'status' => $response->status(),
                            'response' => $response->json()
                        ]);

                        $results[$pluginSlug] = [
                            'success' => false,
                            'message' => $errorMessage
                        ];
                        $failCount++;
                        continue;
                    }

                    // Supprimer le plugin de la base de données
                    $pluginName = $plugin->name;
                    $plugin->delete();

                    Log::info('Plugin deleted successfully in bulk operation', [
                        'plugin_name' => $pluginName,
                        'plugin_slug' => $pluginSlug
                    ]);

                    $results[$pluginSlug] = [
                        'success' => true,
                        'message' => 'Plugin deleted successfully',
                        'name' => $pluginName
                    ];
                    $successCount++;

                    // Petite pause entre chaque suppression pour éviter de surcharger l'API
                    usleep(500000); // 500ms

                } catch (\Exception $e) {
                    Log::error('Error deleting plugin in bulk operation', [
                        'plugin_slug' => $pluginSlug,
                        'error' => $e->getMessage()
                    ]);

                    $results[$pluginSlug] = [
                        'success' => false,
                        'message' => $e->getMessage()
                    ];
                    $failCount++;
                }
            }

            Log::info('Bulk plugin deletion completed', [
                'total' => count($pluginSlugs),
                'success' => $successCount,
                'failed' => $failCount
            ]);

            return response()->json([
                'success' => true,
                'data' => [
                    'results' => $results,
                    'total' => count($pluginSlugs),
                    'success' => $successCount,
                    'failed' => $failCount
                ],
                'message' => $successCount > 0
                    ? ($failCount > 0
                        ? "Successfully deleted {$successCount} plugins, {$failCount} plugin(s) failed"
                        : "Successfully deleted all {$successCount} plugins")
                    : "Failed to delete any plugins"
            ]);

        } catch (\Exception $e) {
            Log::error('Error in bulk plugin deletion', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error deleting plugins: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get plugin delete arguments
     *
     * @return array
     */
    private function get_plugin_delete_args() {
        return array(
            'plugin_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_plugin_exists'),
            ),
        );
    }

    /**
     * Get plugin update arguments
     *
     * @return array
     */
    private function get_plugin_update_args() {
        return array(
            'plugin_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_plugin_exists'),
            ),
        );
    }

    /**
     * Get bulk update arguments
     *
     * @return array
     */
    private function get_bulk_update_args() {
        return array(
            'plugins' => array(
                'required'          => true,
                'type'             => 'array',
                'sanitize_callback' => function($plugins) {
                    return array_map('sanitize_text_field', $plugins);
                },
            ),
        );
    }

    /**
     * Get plugin status arguments
     *
     * @return array
     */
    private function get_plugin_status_args() {
        return array(
            'plugin_slug' => array(
                'required'          => true,
                'type'             => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => array($this, 'validate_plugin_exists'),
            ),
            'status' => array(
                'required' => true,
                'type'    => 'string',
                'enum'    => array('active', 'inactive'),
            ),
        );
    }

    /**
     * Validate plugin exists
     *
     * @param string $slug Plugin slug
     * @return bool
     */
    public function validate_plugin_exists($slug) {
        $exists = file_exists(WP_PLUGIN_DIR . '/' . $slug);
        error_log('[NexaWP] Validating plugin exists: ' . $slug . ' - ' . ($exists ? 'true' : 'false'));
        return $exists;
    }

    /**
     * Verify API key
     *
     * @param WP_REST_Request $request Request object
     * @return bool
     */
    public function verify_api_key($request) {
        $api_key_request = $request->get_header('Authorization');
        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');

        error_log('[NexaWP] Verifying API key - Request: ' . substr($api_key_request, -4) . ' Stored: ' . substr($api_key_stored, -4));

        return $api_key_request === $api_key_stored;
    }
}
