<?php
/**
 * NexaWP Core Controller
 *
 * Handles all WordPress core related REST API endpoints
 *
 * @package NexaWP
 */

class NexaWP_Core_Controller {

    /**
     * API Namespace
     *
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Register REST routes for WordPress core
     */
    public function register_routes() {
        error_log('[NexaWP] Registering core routes');

        // Get WordPress core status
        register_rest_route($this->namespace, '/core/status', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_core_status'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Update WordPress core
        register_rest_route($this->namespace, '/core/update', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'update_core'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        // Update auto-update settings
        register_rest_route($this->namespace, '/core/auto-update', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'update_auto_update_settings'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => $this->get_auto_update_args(),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        register_rest_route($this->namespace, '/core/debug', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'update_debug_mode'),
                'permission_callback' => array($this, 'verify_api_key'),
                'args'                => array(
                    'enabled' => array(
                        'required'          => true,
                        'type'              => 'boolean',
                    ),
                ),
            ),
            array(
                'methods'             => 'OPTIONS',
                'callback'            => function() {
                    return new WP_REST_Response(null, 200);
                },
                'permission_callback' => '__return_true',
            )
        ));

        error_log('[NexaWP] Core routes registered');
    }

    /**
     * Get WordPress core status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_core_status($request) {
        error_log('[NexaWP] Starting get_core_status request');

        try {
            global $wp_version;
            require_once ABSPATH . 'wp-admin/includes/update.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

            wp_version_check(); // Check for WordPress updates
            $update_core = get_site_transient('update_core'); // Get update information

            $update_available = false;
            $latest_version = $wp_version;
            $update_type = null;

            if (isset($update_core->updates) && is_array($update_core->updates)) {
                foreach ($update_core->updates as $update) {
                    if (isset($update->response) && $update->response == 'upgrade') {
                        $update_available = true;
                        $latest_version = $update->version;
                        $update_type = isset($update->response) ? $update->response : null;
                        break;
                    }
                }
            }

            // Get auto-update settings
            $auto_updates = array(
                'enabled' => (bool) get_site_option('auto_update_core', false),
                'major' => (bool) get_site_option('allow_major_auto_core_updates', false),
                'minor' => (bool) get_site_option('allow_minor_auto_core_updates', true),
            );

            $core_data = array(
                'version'          => $wp_version,
                'update_available' => $update_available,
                'latest_version'   => $latest_version,
                'update_type'      => $update_type,
                'locale'           => get_locale(),
                'wp_debug'         => defined('WP_DEBUG') && WP_DEBUG,
                'multisite'        => is_multisite(),
                'auto_updates'     => $auto_updates,
                'update_package'   => $update_available ? ($update_core->updates[0]->package ?? null) : null,
                'local_package'    => $update_available ? ($update_core->updates[0]->local_package ?? null) : null,
            );

            error_log('[NexaWP] Successfully prepared core status response');
            return new WP_REST_Response($core_data, 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in get_core_status: ' . $e->getMessage());
            return new WP_Error(
                'core_status_error',
                __('Failed to retrieve core status', 'nexawp-connector'),
                array('status' => 500)
            );
        }
    }

    /**
     * Update WordPress core
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    /**
     * Update (or re-install) WordPress core
     */
    public function update_core($request) {
        error_log('[NexaWP] Starting update_core request');

        try {
            require_once ABSPATH . 'wp-admin/includes/update.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/class-core-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';

            // Force WP to check for updates
            wp_version_check([], true);
            $update_core = get_site_transient('update_core');

            global $wp_version;
            $old_version = $wp_version;

            // Get the first update object if any
            $update = $update_core->updates[0] ?? null;

            // Force re-install logic
            if (!$update) {
                // If no update object, create one from scratch
                $update = (object) [
                    'response' => 'upgrade',
                    'download' => '',
                    'locale'   => get_locale(),
                    'current'  => $wp_version,
                    'version'  => $wp_version,
                    'package'  => '',
                ];
            } else {
                // If there's an update object, override these fields
                $update->response = 'upgrade';
                if (empty($update->package)) {
                    // If you want to re-download a ZIP, set $update->package here
                    $update->package = '';
                }
            }

            error_log('[NexaWP] Forcing core (re-)installation');

            // Prepare upgrader
            $skin = new Automatic_Upgrader_Skin();
            $upgrader = new Core_Upgrader($skin);

            // Maintenance mode
            $maintenance_file = ABSPATH . '.maintenance';
            try {
                error_log('[NexaWP] Activating maintenance mode');
                @file_put_contents($maintenance_file, '<?php $upgrading = ' . time() . '; ?>');
                set_time_limit(300);

                // Perform re-install
                $result = $upgrader->upgrade($update);

                error_log('[NexaWP] Core update completed with result: ' . (is_wp_error($result) ? $result->get_error_message() : 'success'));
            } finally {
                // Always remove maintenance file
                if (file_exists($maintenance_file)) {
                    @unlink($maintenance_file);
                    error_log('[NexaWP] Maintenance mode deactivated');
                }
            }

            if (is_wp_error($result)) {
                error_log('[NexaWP] Core update failed: ' . $result->get_error_message());
                return $result;
            }
            if ($result === false) {
                error_log('[NexaWP] Core update failed with unknown error');
                return new WP_Error(
                    'update_failed',
                    __('WordPress core update failed.', 'nexawp-connector'),
                    ['status' => 500]
                );
            }

            // Re-check version
            global $wp_version;
            $new_version = $wp_version;
            error_log("[NexaWP] Core successfully updated from {$old_version} to {$new_version}");

            // We'll also send 'update_in_progress' => false so that Laravel can finalize
            $started_at = current_time('mysql');

            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf(
                    __('WordPress core successfully updated (re-installed) to version %s', 'nexawp-connector'),
                    $new_version
                ),
                'data' => [
                    'old_version'        => $old_version,
                    'new_version'        => $new_version,
                    'updated_at'         => current_time('mysql'),
                    'started_at'         => $started_at,
                    'update_in_progress' => false,
                ]
            ], 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in update_core: ' . $e->getMessage());
            $maintenance_file = ABSPATH . '.maintenance';
            if (file_exists($maintenance_file)) {
                @unlink($maintenance_file);
            }
            return new WP_Error(
                'core_update_error',
                __('Failed to update/re-install WordPress core: ', 'nexawp-connector') . $e->getMessage(),
                ['status' => 500]
            );
        }
    }

    /**
     * Update auto-update settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_auto_update_settings($request) {
        error_log('[NexaWP] Starting update_auto_update_settings request');

        try {
            $settings = $request->get_param('settings');

            if (!is_array($settings)) {
                error_log('[NexaWP] Invalid auto-update settings');
                return new WP_Error(
                    'invalid_settings',
                    __('Invalid auto-update settings.', 'nexawp-connector'),
                    array('status' => 400)
                );
            }

            // Update auto-update settings
            $enabled = isset($settings['enabled']) ? (bool) $settings['enabled'] : false;
            $major = isset($settings['major']) ? (bool) $settings['major'] : false;
            $minor = isset($settings['minor']) ? (bool) $settings['minor'] : true;

            update_site_option('auto_update_core', $enabled);
            update_site_option('allow_major_auto_core_updates', $major);
            update_site_option('allow_minor_auto_core_updates', $minor);

            error_log('[NexaWP] Auto-update settings updated');

            // Return updated settings
            $updated_settings = array(
                'enabled' => (bool) get_site_option('auto_update_core', false),
                'major' => (bool) get_site_option('allow_major_auto_core_updates', false),
                'minor' => (bool) get_site_option('allow_minor_auto_core_updates', true),
            );

            return new WP_REST_Response(array(
                'success' => true,
                'message' => __('Auto-update settings updated successfully.', 'nexawp-connector'),
                'data' => array(
                    'auto_updates' => $updated_settings,
                    'updated_at'   => current_time('mysql')
                )
            ), 200);

        } catch (Exception $e) {
            error_log('[NexaWP] Error in update_auto_update_settings: ' . $e->getMessage());
            return new WP_Error(
                'auto_update_settings_error',
                __('Failed to update auto-update settings: ', 'nexawp-connector') . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Arguments for auto-update settings
     *
     * @return array
     */
    private function get_auto_update_args() {
        return array(
            'settings' => array(
                'required' => true,
                'type'     => 'object',
            ),
        );
    }

    /**
     * Update WP_DEBUG in wp-config.php
     */
    public function update_debug_mode($request) {
        error_log('[NexaWP] Starting update_debug_mode request');

        // 1) Lecture du paramètre "enabled"
        $enabled = $request->get_param('enabled');
        $enabled_bool = (bool) $enabled;

        try {
            // 2) Init Filesystem
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';

            global $wp_filesystem;
            if (!function_exists('WP_Filesystem')) {
                include_once ABSPATH . 'wp-admin/includes/file.php';
            }

            $creds = request_filesystem_credentials('', '', false, false, array());
            if (!WP_Filesystem($creds)) {
                return new WP_Error(
                    'fs_init_error',
                    __('Could not initialize filesystem', 'nexawp-connector'),
                    array('status' => 500)
                );
            }
            // Check if $wp_filesystem is usable
            if (!is_object($wp_filesystem)) {
                return new WP_Error(
                    'fs_unavailable',
                    __('Filesystem is not available', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            // 3) On lit wp-config.php
            $config_path = ABSPATH . 'wp-config.php';
            if (!$wp_filesystem->exists($config_path)) {
                return new WP_Error(
                    'config_missing',
                    __('Could not find wp-config.php', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            $config_contents = $wp_filesystem->get_contents($config_path);
            if (!$config_contents) {
                return new WP_Error(
                    'config_unreadable',
                    __('Could not read wp-config.php', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            // 4) Remplacer la ligne define( 'WP_DEBUG', ... );
            $debug_pattern = "/define\s*\(\s*('|\")WP_DEBUG('|\")\s*,\s*(true|false)\s*\)\s*;/i";

            $new_define = "define('WP_DEBUG', " . ($enabled_bool ? 'true' : 'false') . ");";
            $updated_contents = preg_replace($debug_pattern, $new_define, $config_contents, -1, $count);

            // Si $count == 0 => on n'a pas trouvé la ligne existante, on peut l'ajouter
            if ($count === 0) {
                // Ajout en fin de fichier, par exemple
                $updated_contents .= "\n\n/** NexaWP auto-added WP_DEBUG toggle **/\n" . $new_define . "\n";
            }

            // 5) Écrire le nouveau contenu
            if (!$wp_filesystem->put_contents($config_path, $updated_contents, FS_CHMOD_FILE)) {
                return new WP_Error(
                    'config_write_failed',
                    __('Could not write changes to wp-config.php', 'nexawp-connector'),
                    array('status' => 500)
                );
            }

            // 6) Réponse
            return new WP_REST_Response(array(
                'success' => true,
                'message' => sprintf(
                    __('WP_DEBUG successfully set to %s', 'nexawp-connector'),
                    $enabled_bool ? 'true' : 'false'
                ),
                'data' => array(
                    'wp_debug' => $enabled_bool,
                )
            ), 200);

        } catch (\Exception $e) {
            error_log('[NexaWP] Error in update_debug_mode: ' . $e->getMessage());

            return new WP_Error(
                'debug_toggle_error',
                __('Failed to toggle WP_DEBUG: ', 'nexawp-connector') . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Verify API key
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function verify_api_key($request) {
        $api_key_request = $request->get_header('Authorization');
        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');

        error_log('[NexaWP] Verifying API key for core controller');

        return $api_key_request === $api_key_stored;
    }
}
