<?php
/**
 * NexaWP Backup Controller
 *
 * Handles all backup-related REST API endpoints
 *
 * @package NexaWP
 */

class NexaWP_Backup_Controller {

    /**
     * API Namespace
     *
     * @var string
     */
    private $namespace = 'nexawp/v1';

    /**
     * Backup directory path
     *
     * @var string
     */
    private $backup_dir;

    /**
     * Constructor
     */
    public function __construct() {
        $this->backup_dir = WP_CONTENT_DIR . '/nexawp-backups/';
        $this->ensure_backup_directory();
    }

    /**
     * Register REST routes for backups
     */
    public function register_routes() {
        // Create backup
        register_rest_route($this->namespace, '/backups', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'create_backup'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_backup_args(),
        ));

        // Get backup status
        register_rest_route($this->namespace, '/backups/(?P<id>[a-zA-Z0-9-]+)', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_backup_status'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // List backups
        register_rest_route($this->namespace, '/backups', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'list_backups'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Delete backup
        register_rest_route($this->namespace, '/backups/(?P<id>[a-zA-Z0-9-]+)', array(
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => array($this, 'delete_backup'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Schedule backup
        register_rest_route($this->namespace, '/backups/schedule', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'schedule_backup'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args'                => $this->get_schedule_args(),
        ));
    }

    /**
     * Create backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function create_backup($request) {
        global $wpdb;

        $backup_id = uniqid('backup-', true);
        $type = $request->get_param('type') ?: 'full';
        $backup_dir = $this->backup_dir . $backup_id . '/';

        if (!wp_mkdir_p($backup_dir)) {
            return new WP_Error(
                'backup_dir_creation_failed',
                __('Failed to create backup directory', 'nexawp-connector'),
                array('status' => 500)
            );
        }

        // Store backup metadata
        $metadata = array(
            'id'         => $backup_id,
            'type'       => $type,
            'status'     => 'in_progress',
            'started_at' => current_time('mysql'),
            'files'      => array(),
        );

        update_option('nexawp_backup_' . $backup_id, $metadata);

        // Start backup process
        if ($type === 'full' || $type === 'files') {
            $files_result = $this->backup_files($backup_dir);
            if (is_wp_error($files_result)) {
                return $files_result;
            }
            $metadata['files']['filesystem'] = $files_result;
        }

        if ($type === 'full' || $type === 'database') {
            $db_result = $this->backup_database($backup_dir);
            if (is_wp_error($db_result)) {
                return $db_result;
            }
            $metadata['files']['database'] = $db_result;
        }

        // Update backup status
        $metadata['status'] = 'completed';
        $metadata['completed_at'] = current_time('mysql');
        update_option('nexawp_backup_' . $backup_id, $metadata);

        return new WP_REST_Response($metadata, 200);
    }

    /**
     * Backup files
     *
     * @param string $backup_dir Backup directory path
     * @return string|WP_Error Path to backup file or error
     */
    private function backup_files($backup_dir) {
        $filename = 'files-' . date('Y-m-d-H-i-s') . '.zip';
        $filepath = $backup_dir . $filename;

        // Create ZIP archive
        $zip = new ZipArchive();
        if ($zip->open($filepath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return new WP_Error(
                'zip_creation_failed',
                __('Failed to create ZIP archive', 'nexawp-connector'),
                array('status' => 500)
            );
        }

        // Add files to ZIP
        $root_dir = ABSPATH;
        $exclude_dirs = array('wp-content/nexawp-backups');

        $iterator = new RecursiveDirectoryIterator($root_dir);
        $files = new RecursiveIteratorIterator($iterator);

        foreach ($files as $file) {
            if ($file->isFile()) {
                $file_path = $file->getRealPath();
                $relative_path = str_replace($root_dir, '', $file_path);

                // Skip excluded directories
                $skip = false;
                foreach ($exclude_dirs as $exclude_dir) {
                    if (strpos($relative_path, $exclude_dir) === 0) {
                        $skip = true;
                        break;
                    }
                }

                if (!$skip) {
                    $zip->addFile($file_path, $relative_path);
                }
            }
        }

        $zip->close();

        return $filename;
    }

    /**
     * Backup database
     *
     * @param string $backup_dir Backup directory path
     * @return string|WP_Error Path to backup file or error
     */
    private function backup_database($backup_dir) {
        global $wpdb;

        $filename = 'database-' . date('Y-m-d-H-i-s') . '.sql';
        $filepath = $backup_dir . $filename;

        $tables = $wpdb->get_results('SHOW TABLES', ARRAY_N);
        $output = '';

        foreach ($tables as $table) {
            $table_name = $table[0];

            // Get create table syntax
            $create_table = $wpdb->get_row("SHOW CREATE TABLE `$table_name`", ARRAY_N);
            $output .= "\n\n" . $create_table[1] . ";\n\n";

            // Get table data
            $rows = $wpdb->get_results("SELECT * FROM `$table_name`", ARRAY_A);
            foreach ($rows as $row) {
                $values = array_map(array($wpdb, '_real_escape'), $row);
                $output .= "INSERT INTO `$table_name` VALUES ('" . implode("','", $values) . "');\n";
            }
        }

        if (file_put_contents($filepath, $output) === false) {
            return new WP_Error(
                'db_backup_failed',
                __('Failed to write database backup', 'nexawp-connector'),
                array('status' => 500)
            );
        }

        return $filename;
    }

    /**
     * Get backup status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_backup_status($request) {
        $backup_id = $request->get_param('id');
        $metadata = get_option('nexawp_backup_' . $backup_id);

        if (!$metadata) {
            return new WP_Error(
                'backup_not_found',
                __('Backup not found', 'nexawp-connector'),
                array('status' => 404)
            );
        }

        return new WP_REST_Response($metadata, 200);
    }

    /**
     * List backups
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function list_backups($request) {
        global $wpdb;

        $backups = array();
        $options = $wpdb->get_results(
            "SELECT option_name, option_value 
            FROM $wpdb->options 
            WHERE option_name LIKE 'nexawp_backup_%'",
            ARRAY_A
        );

        foreach ($options as $option) {
            $backups[] = maybe_unserialize($option['option_value']);
        }

        return new WP_REST_Response($backups, 200);
    }

    /**
     * Delete backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_backup($request) {
        $backup_id = $request->get_param('id');
        $metadata = get_option('nexawp_backup_' . $backup_id);

        if (!$metadata) {
            return new WP_Error(
                'backup_not_found',
                __('Backup not found', 'nexawp-connector'),
                array('status' => 404)
            );
        }

        // Delete backup files
        $backup_dir = $this->backup_dir . $backup_id;
        if (file_exists($backup_dir)) {
            $this->delete_directory($backup_dir);
        }

        // Delete backup metadata
        delete_option('nexawp_backup_' . $backup_id);

        return new WP_REST_Response(array(
            'message' => __('Backup deleted successfully', 'nexawp-connector')
        ), 200);
    }

    /**
     * Schedule backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function schedule_backup($request) {
        $frequency = $request->get_param('frequency');
        $type = $request->get_param('type') ?: 'full';

        $schedule = array(
            'frequency' => $frequency,
            'type'      => $type,
            'next_run'  => $this->calculate_next_run($frequency),
        );

        update_option('nexawp_backup_schedule', $schedule);

        // Schedule the event
        if (!wp_next_scheduled('nexawp_scheduled_backup')) {
            wp_schedule_event(time(), $frequency, 'nexawp_scheduled_backup');
        }

        return new WP_REST_Response($schedule, 200);
    }

    /**
     * Calculate next backup run time
     *
     * @param string $frequency Backup frequency
     * @return int Timestamp
     */
    private function calculate_next_run($frequency) {
        $now = time();

        switch ($frequency) {
            case 'daily':
                return strtotime('tomorrow', $now);
            case 'weekly':
                return strtotime('next week', $now);
            case 'monthly':
                return strtotime('first day of next month', $now);
            default:
                return $now;
        }
    }

    /**
     * Ensure backup directory exists and is writable
     */
    private function ensure_backup_directory() {
        if (!file_exists($this->backup_dir)) {
            wp_mkdir_p($this->backup_dir);
        }

        // Create .htaccess to protect backup directory
        $htaccess = $this->backup_dir . '.htaccess';
        if (!file_exists($htaccess)) {
            file_put_contents($htaccess, "deny from all");
        }

        // Create index.php to prevent directory listing
        $index = $this->backup_dir . 'index.php';
        if (!file_exists($index)) {
            file_put_contents($index, "<?php\n// Silence is golden.");
        }
    }

    /**
     * Delete directory and its contents
     *
     * @param string $dir Directory path
     */
    private function delete_directory($dir) {
        if (!file_exists($dir)) {
            return;
        }

        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->delete_directory($path) : unlink($path);
        }

        return rmdir($dir);
    }

    /**
     * Verify API key
     *
     * @param WP_REST_Request $request Request object
     * @return bool
     */
    public function verify_api_key($request) {
        $api_key_request = $request->get_header('Authorization');
        $api_key_stored = 'Bearer ' . get_option('nexawp_api_key');
        return $api_key_request === $api_key_stored;
    }

    /**
     * Get backup arguments
     *
     * @return array
     */
    private function get_backup_args() {
        return array(
            'type' => array(
                'type'              => 'string',
                'default'           => 'full',
                'enum'              => array('full', 'database', 'files'),
                'sanitize_callback' => 'sanitize_text_field',
            ),
        );
    }

    /**
     * Get schedule arguments
     *
     * @return array
     */
    private function get_schedule_args() {
        return array(
            'frequency' => array(
                'required'          => true,
                'type'             => 'string',
                'enum'             => array('daily', 'weekly', 'monthly'),
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'type' => array(
                'type'              => 'string',
                'default'           => 'full',
                'enum'              => array('full', 'database', 'files'),
                'sanitize_callback' => 'sanitize_text_field',
            ),
        );
    }
}
